import { DialogClose, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { ComponentWithClassName } from "@/types/utils";
import { TransformedStatewideService } from "../type/transformers";
import { Form, FormControl, FormField, FormItem, FormMessage } from "@/components/ui/form";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { useForm } from "react-hook-form";
import { z } from "zod";
import { zodResolver } from "@hookform/resolvers/zod";
import LoadingCircle from "@/components/icons/loading.circle";
import { deleteStatewideService } from "../lib/actions";
import { revalidateGetStatewideServices } from "@/hooks/use-get-statewide-services";

export function WorkflowDialogDelete({ data, setClose, className }: ComponentWithClassName<{ data: NonNullable<TransformedStatewideService>, setClose: () => void }>) {
  const confirmation = data.alias ?? "delete"

  const formSchema = z.object({
    confirmation_field: z.string()
      .refine((password) => password === confirmation, { message: `Form must match '${confirmation}'.` }),
  });

  const form = useForm<z.infer<typeof formSchema>>({
    resolver: zodResolver(formSchema),
  })
  const { formState: { isDirty, isValid, isSubmitting } } = form

  return (
    <DialogContent className={className}>
      <DialogHeader>
        <DialogTitle>Deleting this service?</DialogTitle>
        <DialogDescription>
          Deleting services is permanent. To confirm this deletion type{" "}
          <span className="font-mono rounded-md bg-muted px-1.5 py-0.5">{confirmation}</span>{" "}
          in the form.
        </DialogDescription>
      </DialogHeader>

      <Form {...form}>
        <form
          onSubmit={form.handleSubmit(async (values) => {
            if (!data.id || confirmation !== values.confirmation_field) {
              setClose()
              return;
            }

            await deleteStatewideService(data.id)
            await revalidateGetStatewideServices()

            setClose()
          })}
          className="space-y-8"
        >

          <FormField
            control={form.control}
            name="confirmation_field"
            render={({ field }) => (
              <FormItem>
                <FormControl>
                  <Input
                    placeholder={confirmation}
                    type="text"
                    {...field} />
                </FormControl>
                {/* <FormDescription>This is your system administrative email address.</FormDescription> */}
                <FormMessage />
              </FormItem>
            )}
          />

          <div className="flex flex-row gap-2 justify-end">
            <DialogClose asChild>
              <Button variant={"secondary"}>Cancel</Button>
            </DialogClose>

            <Button type="submit" variant={"destructive"} disabled={!isDirty || !isValid || isSubmitting}>
              Delete service{" "}
              {isSubmitting && <LoadingCircle className="ml-2" />}
            </Button>
          </div>
        </form>
      </Form>
    </DialogContent>
  )
}
